/*** /
 
 This file is part of Golly, a Game of Life Simulator.
 Copyright (C) 2013 Andrew Trevorrow and Tomas Rokicki.
 
 This program is free software; you can redistribute it and/or
 modify it under the terms of the GNU General Public License
 as published by the Free Software Foundation; either version 2
 of the License, or (at your option) any later version.
 
 This program is distributed in the hope that it will be useful,
 but WITHOUT ANY WARRANTY; without even the implied warranty of
 MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 GNU General Public License for more details.
 
 You should have received a copy of the GNU General Public License
 along with this program; if not, write to the Free Software
 Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA.
 
 Web site:  http://sourceforge.net/projects/golly
 Authors:   rokicki@gmail.com  andrew@trevorrow.com
 
 / ***/

// for opening zip files and extracting files within them
// (much thanks to the http://code.google.com/p/objective-zip/ project)
#import "Objective-Zip/ZipFile.h"
#import "Objective-Zip/ZipException.h"
#import "Objective-Zip/FileInZipInfo.h"
#import "Objective-Zip/ZipReadStream.h"

#import "GollyAppDelegate.h"        // for SwitchToPatternTab
#import "HelpViewController.h"      // for ShowHelp
#import "InfoViewController.h"      // for ShowTextFile

#include <string>           // for std::string
#include <list>             // for std::list

#include "bigint.h"
#include "lifealgo.h"
#include "qlifealgo.h"
#include "hlifealgo.h"
#include "readpattern.h"    // for readpattern
#include "writepattern.h"   // for writepattern, pattern_format

#include "utils.h"          // for Warning
#include "prefs.h"          // for SavePrefs, allowundo, userrules, etc
#include "status.h"         // for SetMessage
#include "algos.h"          // for CreateNewUniverse, algo_type, algoinfo, etc
#include "layer.h"          // for currlayer, etc
#include "control.h"        // for SetGenIncrement, ChangeAlgorithm
#include "view.h"           // for origin_restored, nopattupdate
#include "undo.h"           // for UndoRedo
#include "file.h"

// -----------------------------------------------------------------------------

std::string GetBaseName(const char* path)
{
    // extract basename from given path
    std::string basename = path;
    size_t lastsep = basename.rfind('/');
    if (lastsep != std::string::npos) {
        basename = basename.substr(lastsep+1);
    }
    return basename;
}

// -----------------------------------------------------------------------------

void SetPatternTitle(const char* filename)
{
    if ( filename[0] != 0 ) {
        // remember current file name
        currlayer->currname = filename;
        // show currname in current layer's menu item
        //!!! UpdateLayerItem(currindex);
    }
    
    /*!!!??? move this logic into StatusView's drawRect
    std::string prefix = "";
    
    // display asterisk if pattern has been modified
    if (currlayer->dirty) prefix += "*";
    
    int cid = currlayer->cloneid;
    while (cid > 0) {
        // display one or more "=" chars to indicate this is a cloned layer
        prefix += "=";
        cid--;
    }
    
    std::string rule = GetRuleName(currlayer->algo->getrule());
    std::string wtitle;
    wtitle.Printf("%s%s [%s]", prefix.c_str(), currlayer->currname.c_str(), rule.c_str());
    */
}

// -----------------------------------------------------------------------------

void CreateUniverse()
{
    // save current rule
    std::string oldrule = currlayer->algo->getrule();
    
    // delete old universe and create new one of same type
    delete currlayer->algo;
    currlayer->algo = CreateNewUniverse(currlayer->algtype);
    
    // ensure new universe uses same rule (and thus same # of cell states)
    RestoreRule(oldrule.c_str());
    
    // increment has been reset to 1 but that's probably not always desirable
    // so set increment using current step size
    SetGenIncrement();
}

// -----------------------------------------------------------------------------

void NewPattern(const char* title)
{
    if (generating) Warning("Bug detected in NewPattern!");
    
    //!!! if (askonnew && currlayer->dirty && !SaveCurrentLayer()) return;
    
    currlayer->savestart = false;
    currlayer->currfile.clear();
    currlayer->startgen = 0;
    
    // reset step size before CreateUniverse calls SetGenIncrement
    currlayer->currbase = algoinfo[currlayer->algtype]->defbase;
    currlayer->currexpo = 0;
    
    // create new, empty universe of same type and using same rule
    CreateUniverse();
    
    // clear all undo/redo history
    currlayer->undoredo->ClearUndoRedo();
    
    // possibly clear selection
    currlayer->currsel.Deselect();
    
    // initially in drawing mode
    currlayer->touchmode = drawmode;
    
    // reset location and scale
    currlayer->view->setpositionmag(bigint::zero, bigint::zero, newmag);

    // best to restore true origin
    if (currlayer->originx != bigint::zero || currlayer->originy != bigint::zero) {
        currlayer->originx = 0;
        currlayer->originy = 0;
        SetMessage(origin_restored);
    }
    
    // restore default colors for current algo/rule
    UpdateLayerColors();
    
    MarkLayerClean(title);     // calls SetPatternTitle
}

// -----------------------------------------------------------------------------

bool LoadPattern(const char* path, const char* newtitle)
{
    if ( !FileExists(path) ) {
        std::string msg = "The file does not exist:\n";
        msg += path;
        Warning(msg.c_str());
        return false;
    }
    
    // newtitle is only empty if called from ResetPattern/RestorePattern
    if (newtitle[0] != 0) {
        //!!! if (askonload && currlayer->dirty && !SaveCurrentLayer()) return false;
        
        currlayer->savestart = false;
        currlayer->currfile = path;
        
        // reset step size
        currlayer->currbase = algoinfo[currlayer->algtype]->defbase;
        currlayer->currexpo = 0;
        
        // clear all undo/redo history
        currlayer->undoredo->ClearUndoRedo();
    }

    // disable pattern update so we see gen=0 and pop=0;
    // in particular, it avoids getPopulation being called which would slow down macrocell loading
    nopattupdate = true;
    
    // save current algo and rule
    algo_type oldalgo = currlayer->algtype;
    std::string oldrule = currlayer->algo->getrule();
    
    // delete old universe and create new one of same type
    delete currlayer->algo;
    currlayer->algo = CreateNewUniverse(currlayer->algtype);
    
    const char* err = readpattern(path, *currlayer->algo);
    if (err) {
        // cycle thru all other algos until readpattern succeeds
        for (int i = 0; i < NumAlgos(); i++) {
            if (i != oldalgo) {
                currlayer->algtype = i;
                delete currlayer->algo;
                currlayer->algo = CreateNewUniverse(currlayer->algtype);
                // readpattern will call setrule
                err = readpattern(path, *currlayer->algo);
                if (!err) break;
            }
        }
        if (err) {
            // no algo could read pattern so restore original algo and rule
            currlayer->algtype = oldalgo;
            delete currlayer->algo;
            currlayer->algo = CreateNewUniverse(currlayer->algtype);
            RestoreRule(oldrule.c_str());
            // Warning(err);
            // current error and original error are not necessarily meaningful
            // so report a more generic error
            Warning("File could not be loaded by any algorithm\n(probably due to an unknown rule).");
        }
    }
    
    // enable pattern update
    nopattupdate = false;
    
    if (newtitle[0] != 0) {
        MarkLayerClean(newtitle);     // calls SetPatternTitle
        
        // restore default base step for current algo
        // (currlayer->currexpo was set to 0 above)
        currlayer->currbase = algoinfo[currlayer->algtype]->defbase;

        SetGenIncrement();
        
        // restore default colors for current algo/rule
        UpdateLayerColors();
        
        currlayer->currsel.Deselect();

        // initially in moving mode
        currlayer->touchmode = movemode;

        currlayer->algo->fit(*currlayer->view, 1);
        currlayer->startgen = currlayer->algo->getGeneration();     // might be > 0
        
        UpdateEverything();
    }
    
    return err == NULL;
}

// -----------------------------------------------------------------------------

void AddRecentPattern(const char* inpath)
{
    std::string path = inpath;
    
    if (path.find(gollydir) == 0) {
        // remove gollydir from start of path
        path.erase(0, gollydir.length());
    }
    
    // check if path is already in recentpatterns
    if (!recentpatterns.empty()) {
        std::list<std::string>::iterator next = recentpatterns.begin();
        while (next != recentpatterns.end()) {
            std::string nextpath = *next;
            if (path == nextpath) {
                if (next == recentpatterns.begin()) {
                    // path is in recentpatterns and at top, so we're done
                    return;
                }
                // remove this path from recentpatterns (we'll add it below)
                recentpatterns.erase(next);
                numpatterns--;
                break;
            }
            next++;
        }
    }
    
    // put given path at start of recentpatterns
    recentpatterns.push_front(path);
    if (numpatterns < maxpatterns) {
        numpatterns++;
    } else {
        // remove the path at end of recentpatterns
        recentpatterns.pop_back();
    }
}

// -----------------------------------------------------------------------------

bool CopyTextToClipboard(const char* text)
{
    UIPasteboard *pboard = [UIPasteboard generalPasteboard];
    pboard.string = [NSString stringWithCString:text encoding:NSUTF8StringEncoding];
    return true;
}

// -----------------------------------------------------------------------------

bool GetTextFromClipboard(std::string& text)
{
    UIPasteboard *pboard = [UIPasteboard generalPasteboard];
    NSString *str = pboard.string;
    if (str == nil) {
        ErrorMessage("No text in pasteboard.");
        return false;
    } else {
        text = [str cStringUsingEncoding:NSUTF8StringEncoding];
        return true;
    }
}

// -----------------------------------------------------------------------------

void LoadRule(const std::string& rulestring)
{
    // load recently installed .rule file
    std::string oldrule = currlayer->algo->getrule();
    int oldmaxstate = currlayer->algo->NumCellStates() - 1;
    
    // selection might change if grid becomes smaller,
    // so save current selection for RememberRuleChange/RememberAlgoChange
    SaveCurrentSelection();
    
    const char* err = currlayer->algo->setrule(rulestring.c_str());
    if (err) {
        // try to find another algorithm that supports the given rule
        for (int i = 0; i < NumAlgos(); i++) {
            if (i != currlayer->algtype) {
                lifealgo* tempalgo = CreateNewUniverse(i);
                err = tempalgo->setrule(rulestring.c_str());
                delete tempalgo;
                if (!err) {
                    // change the current algorithm and switch to the new rule
                    ChangeAlgorithm(i, rulestring.c_str());
                    if (i != currlayer->algtype) {
                        RestoreRule(oldrule.c_str());
                        Warning("Algorithm could not be changed (pattern is too big to convert).");
                    }
                    return;
                }
            }
        }
        // should only get here if rule/table/tree file contains some sort of error
        RestoreRule(oldrule.c_str());
        std::string msg = "Rule is not valid in any algorithm: " + rulestring;
        Warning(msg.c_str());
        return;
    }
    
    std::string newrule = currlayer->algo->getrule();
    if (oldrule != newrule) {
        // if grid is bounded then remove any live cells outside grid edges
        if (currlayer->algo->gridwd > 0 || currlayer->algo->gridht > 0) {
            ClearOutsideGrid();
        }
    }
    
    // new rule might have changed the number of cell states;
    // if there are fewer states then pattern might change
    int newmaxstate = currlayer->algo->NumCellStates() - 1;
    if (newmaxstate < oldmaxstate && !currlayer->algo->isEmpty()) {
        ReduceCellStates(newmaxstate);
    }
    
    // set colors for new rule (loads any .rule/colors/icons file)
    UpdateLayerColors();
    
    if (oldrule != newrule) {
        if (allowundo && !currlayer->stayclean) {
            currlayer->undoredo->RememberRuleChange(oldrule.c_str());
        }
    }
}

// -----------------------------------------------------------------------------

bool ExtractZipEntry(const std::string& zippath, const std::string& entryname, const std::string& outfile)
{
    bool found = false;
    bool ok = false;
	
    @try {
        NSString *nspath = [NSString stringWithCString:zippath.c_str() encoding:NSUTF8StringEncoding];
        ZipFile *zfile= [[ZipFile alloc] initWithFileName:nspath mode:ZipFileModeUnzip];
        
        [zfile goToFirstFileInZip];
        do {
            FileInZipInfo *info = [zfile getCurrentFileInZipInfo];
            std::string thisname = [info.name cStringUsingEncoding:NSUTF8StringEncoding];
            if (thisname == entryname) {
                // we've found the desired entry so copy entry data to given outfile
                ZipReadStream *zipstream = [zfile readCurrentFileInZip];
                NSMutableData *zipdata = [[NSMutableData alloc] initWithLength:info.length];
                int bytesRead = [zipstream readDataWithBuffer:zipdata];
                [zipstream finishedReading];
                if (info.length == 0) {
                    Warning("Zip entry is empty!");
                } else if (bytesRead == info.length) {
                    // write zipdata to outfile
                    FILE* f = fopen(outfile.c_str(), "wb");
                    if (f) {
                        if (fwrite([zipdata mutableBytes], 1, bytesRead, f) != bytesRead) {
                            Warning("Could not write data for zip entry!");
                        } else {
                            ok = true;
                        }
                        fclose(f);
                    } else {
                        Warning("Could not create file for zip entry!");
                    }
                } else {
                    Warning("Failed to read all bytes of zip entry!");
                }
                zipdata = nil;
                found = true;
                break;
            }
        } while ([zfile goToNextFileInZip]);
        
        [zfile close];
        zfile = nil;
        
	} @catch (ZipException *ze) {
        NSString *msg = [NSString stringWithFormat:@"Zip file error: %d - %@", ze.error, [ze reason]]; 
		Warning([msg cStringUsingEncoding:NSUTF8StringEncoding]);
        
	} @catch (id e) {
        NSString *msg = [NSString stringWithFormat:@"Exception caught: %@ - %@", [[e class] description], [e description]]; 
		Warning([msg cStringUsingEncoding:NSUTF8StringEncoding]);
	}
    
    if (found && ok) return true;

    if (!found) {
        std::string msg = "Could not find zip file entry:\n" + entryname;
        Warning(msg.c_str());
    } else {
        // file is probably incomplete so best to delete it
        if (FileExists(outfile)) RemoveFile(outfile);
    }
    
    return false;
}

// -----------------------------------------------------------------------------

void UnzipFile(const std::string& zippath, const std::string& entry)
{
    std::string filename = GetBaseName(entry.c_str());
    std::string tempfile = tempdir + filename;
    
    if ( IsRuleFile(filename) ) {
        // rule-related file should have already been extracted and installed
        // into userrules, so check that file exists and load rule
        std::string rulefile = userrules + filename;
        if (FileExists(rulefile)) {
            // load corresponding rule
            SwitchToPatternTab();
            LoadRule(filename.substr(0, filename.rfind('.')));
        } else {
            std::string msg = "Rule-related file was not installed:\n" + rulefile;
            Warning(msg.c_str());
        }
        
    } else if ( ExtractZipEntry(zippath, entry, tempfile) ) {
        if ( IsHTMLFile(filename) ) {
            // display html file
            ShowHelp(tempfile.c_str());
            
        } else if ( IsTextFile(filename) ) {
            // display text file
            ShowTextFile(tempfile.c_str());
        
        } else if ( IsScriptFile(filename) ) {
            // run script depending on safety check; note that because the script is
            // included in a zip file we don't remember it in the Run Recent submenu
            //!!! CheckBeforeRunning(tempfile, false, zippath);
            Warning("This version of Golly cannot run scripts.");
        
        } else {
            // open pattern but don't remember in recentpatterns
            OpenFile(tempfile.c_str(), false);
        }
    }
}

// -----------------------------------------------------------------------------

static bool RuleInstalled(ZipFile* zipfile, FileInZipInfo* info, const std::string& outfile)
{
    bool ok = true;
    ZipReadStream *zipstream = [zipfile readCurrentFileInZip];
    NSMutableData *zipdata = [[NSMutableData alloc] initWithLength:info.length];
    int bytesRead = [zipstream readDataWithBuffer:zipdata];
    [zipstream finishedReading];
    if (info.length == 0) {
        Warning("Zip entry is empty!");
        ok = false;
    } else if (bytesRead == info.length) {
        // write zipdata to outfile
        FILE* f = fopen(outfile.c_str(), "wb");
        if (f) {
            if (fwrite([zipdata mutableBytes], 1, bytesRead, f) != bytesRead) {
                Warning("Could not write data for zip entry!");
                ok = false;
            }
            fclose(f);
        } else {
            Warning("Could not create file for zip entry!");
            ok = false;
        }
    } else {
        Warning("Failed to read all bytes of zip entry!");
        ok = false;
    }
    zipdata = nil;
    return ok;
}

// -----------------------------------------------------------------------------

void OpenZipFile(const char* zippath)
{
    // Process given zip file in the following manner:
    // - If it contains any .rule files then extract and install those files
    //   into userrules (the user's rules directory).
    // - Build a temporary html file with clickable links to each file entry
    //   and show it in the Help tab.
    
    const std::string indent = "&nbsp;&nbsp;&nbsp;&nbsp;";
    bool dirseen = false;
    bool diffdirs = (userrules != rulesdir);
    std::string firstdir = "";
    std::string lastpattern = "";
    std::string lastscript = "";
    int patternseps = 0;                // # of separators in lastpattern
    int scriptseps = 0;                 // # of separators in lastscript
    int patternfiles = 0;
    int scriptfiles = 0;
    int textfiles = 0;                  // includes html files
    int rulefiles = 0;
    int deprecated = 0;                 // # of .table/tree/colors/icons files
    std::list<std::string> deplist;     // list of installed deprecated files
    std::list<std::string> rulelist;    // list of installed .rule files
    
    // strip off patternsdir or gollydir
    std::string relpath = zippath;
    size_t pos = relpath.find(patternsdir);
    if (pos == 0) {
        relpath.erase(0, patternsdir.length());
    } else {
        pos = relpath.find(gollydir);
        if (pos == 0) relpath.erase(0, gollydir.length());
    }
    
    std::string contents = "<html><body bgcolor=\"#FFFFCE\"><font size=+1><b><p>\nContents of ";
    contents += relpath;
    contents += ":<p>\n";
    
	@try {
        NSString *nspath = [NSString stringWithCString:zippath encoding:NSUTF8StringEncoding];
        ZipFile *zfile= [[ZipFile alloc] initWithFileName:nspath mode:ZipFileModeUnzip];
        
        [zfile goToFirstFileInZip];
        do {
            FileInZipInfo *info = [zfile getCurrentFileInZipInfo];
            // NSLog(@"- %@ %@ %d len=%d (%d)", info.name, info.date, info.size, info.length, info.level);
            
            // examine each entry in zip file and build contents string;
            // also install any .rule files
            std::string name = [info.name cStringUsingEncoding:NSUTF8StringEncoding];
            if (name.find("__MACOSX") == 0 || name.rfind(".DS_Store") != std::string::npos) {
                // ignore meta-data stuff in zip file created on Mac
            } else {
                // indent depending on # of separators in name
                unsigned int sepcount = 0;
                unsigned int i = 0;
                unsigned int len = name.length();
                while (i < len) {
                    if (name[i] == '/') sepcount++;
                    i++;
                }
                // check if 1st directory has multiple separators (eg. in jslife.zip)
                if (name[len-1] == '/' && !dirseen && sepcount > 1) {
                    firstdir = name.substr(0, name.find('/'));
                    contents += firstdir;
                    contents += "<br>\n";
                }
                for (i = 1; i < sepcount; i++) contents += indent;
                
                if (name[len-1] == '/') {
                    // remove terminating separator from directory name
                    name = name.substr(0, name.rfind('/'));
                    name = GetBaseName(name.c_str());
                    if (dirseen && name == firstdir) {
                        // ignore dir already output earlier (eg. in jslife.zip)
                    } else {
                        contents += name;
                        contents += "<br>\n";
                    }
                    dirseen = true;
                    
                } else {
                    // entry is for some sort of file
                    std::string filename = GetBaseName(name.c_str());
                    if (dirseen) contents += indent;
                
                    if ( IsRuleFile(filename) && filename.rfind(".rule") == std::string::npos ) {
                        // this is a deprecated .table/tree/colors/icons file
                        contents += filename;
                        contents += indent;
                        contents += "[deprecated]";
                        deprecated++;
                        // install it into userrules so it can be used below to create a .rule file
                        std::string outfile = userrules + filename;
                        if (RuleInstalled(zfile, info, outfile)) {
                            deplist.push_back(filename);
                        } else {
                            contents += indent;
                            contents += "INSTALL FAILED!";
                        }
                    
                    } else {
                        // user can extract file via special "unzip:" link
                        contents += "<a href=\"unzip:";
                        contents += zippath;
                        contents += ":";
                        contents += name;
                        contents += "\">";
                        contents += filename;
                        contents += "</a>";
                        
                        if ( IsRuleFile(filename) ) {
                            // extract and install .rule file into userrules
                            std::string outfile = userrules + filename;
                            if (RuleInstalled(zfile, info, outfile)) {
                                // file successfully installed
                                rulelist.push_back(filename);
                                contents += indent;
                                contents += "[installed]";
                                if (diffdirs) {
                                    // check if this file overrides similarly named file in rulesdir
                                    std::string clashfile = rulesdir + filename;
                                    if (FileExists(clashfile)) {
                                        contents += indent;
                                        contents += "(overrides file in Rules folder)";
                                    }
                                }
                            } else {
                                // file could not be installed
                                contents += indent;
                                contents += "[NOT installed]";
                                // file is probably incomplete so best to delete it
                                if (FileExists(outfile)) RemoveFile(outfile);
                            }
                            rulefiles++;
                            
                        } else if ( IsHTMLFile(filename) || IsTextFile(filename) ) {
                            textfiles++;
                            
                        } else if ( IsScriptFile(filename) ) {
                            scriptfiles++;
                            lastscript = name;
                            scriptseps = sepcount;
                        
                        } else {
                            patternfiles++;
                            lastpattern = name;
                            patternseps = sepcount;
                        }
                    }
                    contents += "<br>\n";
                }
            }
            
        } while ([zfile goToNextFileInZip]);
        
        [zfile close];
        zfile = nil;

	} @catch (ZipException *ze) {
        NSString *msg = [NSString stringWithFormat:@"Zip file error: %d - %@", ze.error, [ze reason]]; 
		Warning([msg cStringUsingEncoding:NSUTF8StringEncoding]);
        
	} @catch (id e) {
        NSString *msg = [NSString stringWithFormat:@"Exception caught: %@ - %@", [[e class] description], [e description]]; 
		Warning([msg cStringUsingEncoding:NSUTF8StringEncoding]);
	}
    
    if (rulefiles > 0) {
        relpath = userrules;
        pos = relpath.find(gollydir);
        if (pos == 0) relpath.erase(0, gollydir.length());
        contents += "<p>Files marked as \"[installed]\" have been stored in ";
        contents += relpath;
        contents += ".";
    }
    if (deprecated > 0) {
        std::string newrules = CreateRuleFiles(deplist, rulelist);
        if (newrules.length() > 0) {
            contents += "<p>Files marked as \"[deprecated]\" have been used to create new .rule files:<br>\n";
            contents += newrules;
        }
    }
    contents += "\n</b></font></body></html>";
    
    // NOTE: The desktop version of Golly will load a pattern if it's in a "simple" zip file
    // but for the iPad version it's probably less confusing if the zip file's contents are
    // *always* displayed in the Help tab.  We might change this if script support is added.
    
    // write contents to a unique temporary html file
    std::string htmlfile = CreateTempFileName("zip_contents");
    htmlfile += ".html";
    FILE* f = fopen(htmlfile.c_str(), "w");
    if (f) {
        if (fputs(contents.c_str(), f) == EOF) {
            fclose(f);
            Warning("Could not write HTML data to temporary file!");
            return;
        }
    } else {
        Warning("Could not create temporary HTML file!");
        return;
    }
    fclose(f);

    // display temporary html file in Help tab
    ShowHelp(htmlfile.c_str());
}

// -----------------------------------------------------------------------------

/*!!!

void OpenClipboard()
{
    // load and view pattern data stored in clipboard
    wxTextDataObject data;
    if (GetTextFromClipboard(&data)) {
        // copy clipboard data to tempstart so we can handle all formats
        // supported by readpattern
        wxFile outfile(currlayer->tempstart, wxFile::write);
        if ( outfile.IsOpened() ) {
            outfile.Write( data.GetText() );
            outfile.Close();
            LoadPattern(currlayer->tempstart, "clipboard");
            // do NOT delete tempstart -- it can be reloaded by ResetPattern
            // or used by ShowPatternInfo
        } else {
            statusptr->ErrorMessage("Could not create tempstart file!");
        }
    }
}

!!!*/

// -----------------------------------------------------------------------------

void OpenFile(const char* path, bool remember)
{
    // convert path to a full path if necessary
    std::string fullpath = path;
    if (path[0] != '/') {
        if (fullpath.find("Patterns/") == 0) {
            // Patterns directory is inside app bundle so prepend gollydir/Golly.app/
            fullpath = gollydir + "Golly.app/" + fullpath;
        } else {
            fullpath = gollydir + fullpath;
        }
    }
    
    if (IsHTMLFile(path)) {
        // show HTML file in Help tab
        ShowHelp(fullpath.c_str());
        return;
    }
    
    if (IsTextFile(path)) {
        // show text file using InfoViewController
        ShowTextFile(fullpath.c_str());
        return;
    }
    
    if (IsScriptFile(path)) {
        // execute script
        /*!!!
        if (remember) AddRecentScript(path);
        RunScript(path);
        */
        Warning("This version of Golly cannot run scripts.");
        return;
    }
    
    if (IsZipFile(path)) {
        // process zip file
        if (remember) AddRecentPattern(path);   // treat zip file like a pattern file
        OpenZipFile(fullpath.c_str());          // must use full path
        return;
    }

    
    if (IsRuleFile(path)) {
        // switch to rule (.rule file must be in rulesdir or userrules)
        SwitchToPatternTab();
        std::string basename = GetBaseName(path);
        LoadRule(basename.substr(0, basename.rfind('.')));
        return;
    }
    
    // anything else is a pattern file
    if (remember) AddRecentPattern(path);
    std::string basename = GetBaseName(path);
    // best to switch to Pattern tab first in case progress view appears
    SwitchToPatternTab();
    LoadPattern(fullpath.c_str(), basename.c_str());
}

// -----------------------------------------------------------------------------

const char* WritePattern(const char* path,
                         pattern_format format,
                         output_compression compression,
                         int top, int left, int bottom, int right)
{
    // if the format is RLE_format and the grid is bounded then force XRLE_format so that
    // position info is recorded (this position will be used when the file is read)
    if (format == RLE_format && (currlayer->algo->gridwd > 0 || currlayer->algo->gridht > 0))
        format = XRLE_format;
    const char* err = writepattern(path, *currlayer->algo, format,
                                   compression, top, left, bottom, right);
    return err;
}

// -----------------------------------------------------------------------------

void SaveSucceeded(const std::string& path)
{
    // save old info for RememberNameChange
    std::string oldname = currlayer->currname;
    std::string oldfile = currlayer->currfile;
    bool oldsave = currlayer->savestart;
    bool olddirty = currlayer->dirty;
    
    //!!! if (allowundo && !currlayer->stayclean) SavePendingChanges();
    
    if ( currlayer->algo->getGeneration() == currlayer->startgen ) {
        // no need to save starting pattern (ResetPattern can load currfile)
        currlayer->currfile = path;
        currlayer->savestart = false;
    }
    
    // set dirty flag false and update currlayer->currname
    std::string basename = GetBaseName(path.c_str());
    MarkLayerClean(basename.c_str());
    
    if (allowundo && !currlayer->stayclean) {
        currlayer->undoredo->RememberNameChange(oldname.c_str(), oldfile.c_str(), oldsave, olddirty);
    }
}

// -----------------------------------------------------------------------------

bool SavePattern(const std::string& path, pattern_format format, output_compression compression)
{
    bigint top, left, bottom, right;
    int itop, ileft, ibottom, iright;
    currlayer->algo->findedges(&top, &left, &bottom, &right);
    
    if (currlayer->algo->hyperCapable()) {
        // algorithm uses hashlife
        if ( OutsideLimits(top, left, bottom, right) ) {
            // too big so only allow saving as MC file
            if (format != MC_format) {
                Warning("Pattern is outside +/- 10^9 boundary and can't be saved in RLE format.");
                return false;
            }
            itop = ileft = ibottom = iright = 0;
        } else {
            // allow saving as MC or RLE file
            itop = top.toint();
            ileft = left.toint();
            ibottom = bottom.toint();
            iright = right.toint();
        }
    } else {
        // allow saving file only if pattern is small enough
        if ( OutsideLimits(top, left, bottom, right) ) {
            Warning("Pattern is outside +/- 10^9 boundary and can't be saved.");
            return false;
        }
        itop = top.toint();
        ileft = left.toint();
        ibottom = bottom.toint();
        iright = right.toint();
    }
    
    const char* err = WritePattern(path.c_str(), format, compression, itop, ileft, ibottom, iright);
    if (err) {
        Warning(err);
        return false;
    } else {
        std::string msg = "Pattern saved as ";
        msg += GetBaseName(path.c_str());
        DisplayMessage(msg.c_str());
        AddRecentPattern(path.c_str());
        SaveSucceeded(path);
        return true;
    }
}

// -----------------------------------------------------------------------------

bool DownloadFile(const std::string& url, const std::string& filepath)
{
    NSURL *nsurl = [NSURL URLWithString:[NSString stringWithCString:url.c_str() encoding:NSUTF8StringEncoding]];
    if (nsurl == nil) {
        std::string msg = "Bad URL: " + url;
        Warning(msg.c_str());
        return false;
    }
    NSData *urlData = [NSData dataWithContentsOfURL:nsurl];
    if (urlData) {
        [urlData writeToFile:[NSString stringWithCString:filepath.c_str() encoding:NSUTF8StringEncoding] atomically:YES];
        return true;
    } else {
        Warning("Failed to download file!");
        return false;
    }
}

// -----------------------------------------------------------------------------

void GetURL(const std::string& url, const std::string& pageurl)
{
    const char* HTML_PREFIX = "GET-";   // prepended to html filename
    
    std::string fullurl;
    if (url.find("http:") == 0) {
        fullurl = url;
    } else {
        // relative get, so prepend full prefix extracted from pageurl
        std::string urlprefix = GetBaseName(pageurl.c_str());
        // replace HTML_PREFIX with "http://" and convert spaces to '/'
        // (ie. reverse what we do below when building filepath)
        urlprefix.erase(0, strlen(HTML_PREFIX));
        urlprefix = "http://" + urlprefix;
        std::replace(urlprefix.begin(), urlprefix.end(), ' ', '/');
        urlprefix = urlprefix.substr(0, urlprefix.rfind('/')+1);
        fullurl = urlprefix + url;
    }
    
    std::string filename = GetBaseName(fullurl.c_str());
    // remove ugly stuff at start of file names downloaded from ConwayLife.com
    if (filename.find("download.php?f=") == 0 ||
        filename.find("pattern.asp?p=") == 0 ||
        filename.find("script.asp?s=") == 0) {
        filename = filename.substr( filename.find('=')+1 );
    }
    
    // create full path for downloaded file based on given url;
    // first remove initial "http://"
    std::string filepath = fullurl.substr( fullurl.find('/')+1 );
    while (filepath[0] == '/') filepath.erase(0,1);
    if (IsHTMLFile(filename)) {
        // create special name for html file so above code can extract it and set urlprefix
        std::replace(filepath.begin(), filepath.end(), '/', ' ');
        filepath = HTML_PREFIX + filepath;
    } else {
        // no need for url info in file name
        filepath = filename;
    }
    
    if (IsRuleFile(filename)) {
        // create file in user's rules directory
        filepath = userrules + filename;
    } else if (IsHTMLFile(filename)) {
        // nicer to store html files in temporary directory
        filepath = tempdir + filepath;
    } else {
        // all other files are stored in user's download directory
        filepath = downloaddir + filepath;
    }
    
    // download the file and store it in filepath
    if (!DownloadFile(fullurl, filepath)) return;
    
    if (IsHTMLFile(filename)) {
        // display html file in Help tab
        ShowHelp(filepath.c_str());
        
    } else if (IsRuleFile(filename)) {
        // load corresponding rule
        SwitchToPatternTab();
        LoadRule(filename.substr(0, filename.rfind('.')));
        
    } else if (IsTextFile(filename)) {
        // open text file in modal view
        ShowTextFile(filepath.c_str());
    
    } else if (IsScriptFile(filename)) {
        // run script depending on safety check; if it is allowed to run
        // then we remember script in the Run Recent submenu
        //!!! CheckBeforeRunning(filepath, true, wxEmptyString);
        Warning("This version of Golly cannot run scripts.");
    
    } else {
        // assume it's a pattern/zip file, so open it
        OpenFile(filepath.c_str());
    }
}

// -----------------------------------------------------------------------------

void LoadLexiconPattern(const std::string& lexpattern)
{
    // copy lexpattern data to tempstart file
    FILE* f = fopen(currlayer->tempstart.c_str(), "w");
    if (f) {
        if (fputs(lexpattern.c_str(), f) == EOF) {
            fclose(f);
            Warning("Could not write lexicon pattern to tempstart file!");
            return;
        }
    } else {
        Warning("Could not create tempstart file!");
        return;
    }
    fclose(f);
    
    // avoid any pattern conversion (possibly causing ChangeAlgorithm to beep with a message)
    NewPattern();
        
    // all Life Lexicon patterns assume we're using Conway's Life so try
    // switching to B3/S23 or Life; if that fails then switch to QuickLife
    const char* err = currlayer->algo->setrule("B3/S23");
    if (err) {
        // try "Life" in case current algo is RuleLoader and Life.rule/table/tree exists
        err = currlayer->algo->setrule("Life");
    }
    if (err) {
        ChangeAlgorithm(QLIFE_ALGO, "B3/S23");
    }
    
    // load lexicon pattern
    SwitchToPatternTab();
    LoadPattern(currlayer->tempstart.c_str(), "lexicon");
}
