// Author(s): Jeroen Keiren
// Copyright: see the accompanying file COPYING or copy at
// https://svn.win.tue.nl/trac/MCRL2/browser/trunk/COPYING
//
// Distributed under the Boost Software License, Version 1.0.
// (See accompanying file LICENSE_1_0.txt or copy at
// http://www.boost.org/LICENSE_1_0.txt)
//
/// \file mcrl2/data/detail/prover/solver_type.h
/// \brief Enumeration for the types of solvers

#ifndef MCRL2_DATA_DETAIL_PROVER_SOLVER_TYPE_H
#define MCRL2_DATA_DETAIL_PROVER_SOLVER_TYPE_H

#include <iostream>
#include <string>
#include "mcrl2/utilities/exception.h"

namespace mcrl2
{
namespace data
{
namespace detail
{

/// \brief The enumaration type smt_solver_type enumerates all available SMT solvers.
enum smt_solver_type
{
  solver_type_cvc
};

/// \brief standard conversion from string to solver type
inline
smt_solver_type parse_solver_type(const std::string& s)
{
  if(s == "cvc") return solver_type_cvc;
  else throw mcrl2::runtime_error("unknown solver type " + s);
}

/// \brief standard conversion from stream to solver type
inline
std::istream& operator>>(std::istream& is, smt_solver_type& s)
{
  try
  {
    std::string str;
    is >> str;
    s = parse_solver_type(str);
  }
  catch(mcrl2::runtime_error&)
  {
    is.setstate(std::ios_base::failbit);
  }
  return is;
}

/// \brief standard conversion from solver type to string
inline
std::string print_solver_type(const smt_solver_type s)
{
  switch(s)
  {
    case solver_type_cvc: return "cvc";
    default: throw mcrl2::runtime_error("unknown solver type");
  }
}

/// \brief standard conversion from solvert type to stream
inline std::ostream& operator<<(std::ostream& os,smt_solver_type s)
{
  os << print_solver_type(s);
  return os;
}

/// \brief description of solver type
inline
std::string description(const smt_solver_type s)
{
  switch(s)
  {
    case solver_type_cvc: return "the SMT solver CVC3";
    default: throw mcrl2::runtime_error("unknown solver type");
  }
}

} // namespace detail
} // namespace data
} // namespace mcrl2

#endif // MCRL2_DATA_DETAIL_PROVER_SOLVER_TYPE_H
